
#include "stdafx.h"


LRESULT CALLBACK WndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam);


//////////////////////////////////////
//

HRESULT CSystem::Init(HINSTANCE hInst, LPCSTR szCmdLine)
{
   m_hAppInstance = hInst;
   
   // Get Application and Resource Path
   char szPath[MAX_PATH];
   ::GetModuleFileName(NULL, szPath, MAX_PATH);
   LPSTR p = strrchr(szPath, '\\');
   if( p==NULL ) return E_UNEXPECTED;
   strcpy( m_szAppName, p+1 );
   *(p+1) = '\0';
   strcpy( m_szAppPath, szPath );
   p = strrchr( m_szAppName, '.' );
   if( p!=NULL ) *p='\0';
   strcpy( m_szResPath, szPath );
   strcat( m_szResPath, "res\\" );

   strcpy( szPath, m_szAppPath );
   strcat( szPath, m_szAppName );
   strcat( szPath, ".log" );
   m_fileLog.Create(szPath);
   
   LogText("Creating System...");

   LogText("Creating game window...");
   CreateGameWindow();
   ShowGameWindow();

   ParseCommandLine(szCmdLine);
   return S_OK;
};

HRESULT CSystem::Close()
{
   LogText("Shutting system down...");
   CloseWorld();
   CloseGameWindow();

   LogText("System shutdown complete.");
   m_fileLog.Close();
   return S_OK;
};


//////////////////////////////////////
//

void CSystem::LogText(LPCSTR str, ...)
{
   TCHAR szBuffer[128];
   va_list args;
   va_start(args, str);
   ::wvsprintf(szBuffer, str, args);
   m_fileLog.Write(szBuffer,strlen(szBuffer));
   m_fileLog.Write("\n",1);
};

void CSystem::LogError(LPCSTR str, ...)
{
   TCHAR szBuffer[128];
   va_list args;
   va_start(args, str);
   ::wvsprintf(szBuffer, str, args);
   m_fileLog.Write("ERROR: ",7);
   m_fileLog.Write(szBuffer,strlen(szBuffer));
   m_fileLog.Write("\n",1);
};

HRESULT CSystem::LogError(HRESULT Hr)
{
    char errStr[256];
    if(D3DXERR_CAPSNOTSUPPORTED == Hr) {
        strcpy(errStr, "This device lacks required capabilities.\nCannot initialize engine.\n");
    }
    else {
        D3DXGetErrorString(Hr, 256, errStr);
    }
    LogError(errStr);
    return Hr;
};


//////////////////////////////////////
//

HRESULT CSystem::ParseCommandLine(LPCSTR pstrCmdLine)
{
   LogText("Parsing Command Line...");
   // Switches
   LPSTR pstrUpperCmdLine = strdup(pstrCmdLine);
   strupr(pstrUpperCmdLine);
   if( strstr(pstrUpperCmdLine, "/FS")!=NULL ) m_bFullScreen=true;
   if( strstr(pstrUpperCmdLine, "/D:TL")!=NULL ) _Engine.m_dwHwLevel = D3DX_HWLEVEL_TL;
   if( strstr(pstrUpperCmdLine, "/D:RGB")!=NULL ) _Engine.m_dwHwLevel = D3DX_HWLEVEL_2D;
   if( strstr(pstrUpperCmdLine, "/D:HAL")!=NULL ) _Engine.m_dwHwLevel = D3DX_HWLEVEL_RASTER;
   if( strstr(pstrUpperCmdLine, "/D:REF")!=NULL ) _Engine.m_dwHwLevel = D3DX_HWLEVEL_REFERENCE;
   DELETE(pstrUpperCmdLine);
   // Settings
   if( !m_bFullScreen ) _Engine.m_dwNumBackBuffers = 1;
   return S_OK;
};


//////////////////////////////////////
//

HRESULT CSystem::CreateGameWindow()
{
   if( !m_bFullScreen ) m_hCursor = ::LoadCursor(NULL, IDC_ARROW);

   WNDCLASS wc = { 0 };
   wc.style = 0;
   wc.lpfnWndProc = (WNDPROC)WndProc;
   wc.cbClsExtra = 0;
   wc.cbWndExtra = 0;
   wc.hInstance = m_hAppInstance;
   wc.hIcon = ::LoadIcon(m_hAppInstance, MAKEINTRESOURCE(IDI_APP_ICON));
   wc.hCursor = m_hCursor;
   wc.hbrBackground = (HBRUSH)::GetStockObject(BLACK_BRUSH);
   wc.lpszMenuName = NULL;
   wc.lpszClassName = m_szAppName;
   if( !::RegisterClass(&wc) ) return E_FAIL;

   // Create the main window.
   if( m_bFullScreen ) {
      m_hwnd = ::CreateWindow(
                  m_szAppName, 
                  m_szAppName, 
                  WS_POPUP, 
                  CW_USEDEFAULT, 
                  CW_USEDEFAULT,
                  640, 
                  480, 
                  (HWND)NULL, 
                  (HMENU)NULL, 
                  m_hAppInstance,
                  (LPVOID) NULL);
   }
   else {
      m_hwnd = ::CreateWindow(
                  m_szAppName, 
                  m_szAppName, 
                  WS_OVERLAPPEDWINDOW, 
                  CW_USEDEFAULT, 
                  CW_USEDEFAULT,
                  640, 
                  480, 
                  (HWND)NULL, 
                  (HMENU)NULL, 
                  m_hAppInstance, 
                  (LPVOID)NULL);
   }
   if (!m_hwnd) {
      LogError("Unable to create window.");
      return E_FAIL;
   }
   return S_OK;
};

HRESULT CSystem::CloseGameWindow()
{
   ::DestroyWindow(m_hwnd);
   if( m_hCursor!=NULL ) ::DestroyCursor(m_hCursor);
   return S_OK;
};

HRESULT CSystem::ShowGameWindow()
{
   if( !::IsWindow(m_hwnd) ) return E_FAIL;
   ::ShowWindow(m_hwnd, SW_SHOW);
   ::UpdateWindow(m_hwnd);
   return S_OK;
};

HRESULT CSystem::GameLoop()
{
   // Message loop.
   LogText("\nGo...");
   _Engine.m_bReady = true;
   HRESULT Hr;
   MSG msg;
   while(TRUE) {
      if( ::PeekMessage(&msg, NULL, 0U, 0U, PM_REMOVE) ) {
         if( WM_QUIT==msg.message ) break;
         ::TranslateMessage(&msg);
         ::DispatchMessage(&msg);
      }
      else if( m_bActive ) {
         if( FAILED( Hr = _Engine.Draw() ) ) {
            _Engine.m_bReady = false;
            _System.LogError(Hr);
            ::PostQuitMessage(0);
         }
      }
      else {
         ::WaitMessage();
      }
   }
   LogText("Game Done!\n");
   return S_OK;
};


//////////////////////////////////////
//

LRESULT CALLBACK WndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
   switch( uMsg ) {
   case WM_CLOSE:
      ::PostQuitMessage(0);
      break;

   case WM_SIZE:
      if( _Engine.m_bReady && 
         !_System.m_bFullScreen && 
         LOWORD(lParam) > 0 && HIWORD(lParam) > 0 ) {
         HRESULT Hr;
         if( FAILED( Hr = _Engine.m_pD3DX->Resize(LOWORD(lParam),HIWORD(lParam)) ) ) {
            _Engine.m_bReady = false;
            _System.LogError(Hr);
            ::PostQuitMessage(0);
         }
         D3DXMatrixPerspectiveFov(&_Engine.m_matProjection, D3DXToRadian(60.0f), 3.0f / 4.0f, 0.1f, 100.0f);
         _Engine.m_pD3DDevice->SetTransform(D3DTRANSFORMSTATE_PROJECTION, _Engine.m_matProjection);
      }
      break;

   case WM_KEYDOWN:
      {
         _System.m_bKey[wParam] = true;
         for( int i=0; i<_System.m_nSpectators; i++ ) _System.m_ppSpectators[i]->OnKeyDown(wParam);
      }
      break;

   case WM_KEYUP:
      {
         _System.m_bKey[wParam] = false;
         _System.m_keyPress = wParam;
         for( int i=0; i<_System.m_nSpectators; i++ ) _System.m_ppSpectators[i]->OnKeyUp(wParam);
      }
      break;

   case WM_LBUTTONDOWN:
   case WM_RBUTTONDOWN:
      {
         for( int i=0; i<_System.m_nSpectators; i++ ) _System.m_ppSpectators[i]->OnButtonDown(LOWORD(lParam),HIWORD(wParam),LOWORD(wParam));
      }
      return 0L;

   case WM_LBUTTONUP:
   case WM_RBUTTONUP:
      {
         for( int i=0; i<_System.m_nSpectators; i++ ) _System.m_ppSpectators[i]->OnButtonUp(LOWORD(lParam),HIWORD(wParam),LOWORD(wParam));
      }
      return 0L;

   case WM_MOUSEMOVE:
      {
         for( int i=0; i<_System.m_nSpectators; i++ ) _System.m_ppSpectators[i]->OnMouseMove(HIWORD(lParam),LOWORD(lParam));
      }
      return 0L;

   case WM_ACTIVATEAPP:
      if( (wParam==0) && _System.m_bActive ) {
         _System.m_bActive = false;
         if(_Engine.m_pDD) _Engine.m_pDD->FlipToGDISurface();
         ::RedrawWindow(hwnd, NULL, NULL, RDW_FRAME);
      }
      else if( (wParam!=0) && !_System.m_bActive ) {
         _System.m_bActive = true;
      }
      break;

   case WM_PAINT:
     if( _Engine.m_bReady ) {
       HRESULT Hr;
       PAINTSTRUCT ps;
       ::BeginPaint(hwnd, &ps);
       if( FAILED( Hr = _Engine.Draw() ) ) {
         _Engine.m_bReady = false;
         _System.LogError(Hr);
         ::PostQuitMessage(0);
       }
       ::EndPaint(hwnd, &ps);
       return 0L;
     }
     break;

   case WM_ERASEBKGND:
     return 0L;

   case WM_SETCURSOR:
      if( _System.m_bFullScreen ) ::SetCursor(NULL);
      break;

   case WM_COMMAND:
      {
         for( int i=0; i<_System.m_nSpectators; i++ ) _System.m_ppSpectators[i]->OnCommand(uMsg,wParam,lParam);
      }
      break;

   case WM_APPMSG:
      {
         for( int i=0; i<_System.m_nSpectators; i++ ) _System.m_ppSpectators[i]->OnAppMsg(uMsg,wParam,lParam);
      }
      break;

   default:
      break;
   }
   return ::DefWindowProc(hwnd, uMsg, wParam, lParam);
}

