
#define WIN32_LEAN_AND_MEAN		// Exclude rarely-used stuff from Windows headers
#include <windows.h>
#include <D3DX8.h>
#include "dhSimpleMesh.h"

#pragma comment(lib,"d3d8.lib")

#ifdef _DEBUG
   #pragma comment(lib,"dhEngineD.lib")
   #pragma comment(lib,"d3dx8d.lib")
#else
   #pragma comment(lib,"dhEngine.lib")
   #pragma comment(lib,"d3dx8.lib")
#endif

// Forward declarations for all of our functions, see their definitions for more detail
void FatalError(const char *error_msg);
void ask_fullscreen(void);
LRESULT CALLBACK default_window_proc(HWND hwnd,UINT msg,WPARAM wparam,LPARAM lparam);
void init_window(void);
void kill_window(void);
void init_d3d(void);
void kill_d3d(void);
void init_scene(void);
void kill_scene(void);
void init_light(void);
void message_pump(void);
D3DFORMAT find_16bit_mode(void);
void render(void);
void NOP(HINSTANCE p_prev_instance,LPSTR p_cmd_line,int p_show);

const char g_app_name[]="DirectX 8 Lesson 8";

const int g_width=640;
const int g_height=480;

bool g_fullscreen=true;

HWND g_main_window=NULL;

HINSTANCE g_instance;

bool g_app_done=false;

LPDIRECT3D8 g_D3D=NULL;

IDirect3DDevice8 *g_d3d_device=NULL;


LPDIRECT3DTEXTURE8 g_texture=NULL;

struct my_vertex{
    FLOAT x, y, z;   // The untransformed position for the vertex.
    FLOAT nx,ny,nz;
    DWORD colour;
    FLOAT tu, tv;    // The texture coordinates
};

//A handy little 'macro' for our definition of the vertex.  When we use the vertex data
//we have to tell D3D what data we're passing it.The D3DFVF_XYZ specifies that the vertex
//will have coordinate given in model space. D3DFVF_TEX1 specifies that 1 set of texture
//coordinates will be provided as well.  D3DFVF_NORMAL indicates the presence of a normal
//for each vertex. D3DFVF_DIFFUSE indicates that a diffuse colour is part of the vertex as
//well, when lighting is turned off, the DIFFUSE colour will be used for lighting.
#define D3D8T_CUSTOMVERTEX (D3DFVF_XYZ|D3DFVF_TEX1|D3DFVF_NORMAL|D3DFVF_DIFFUSE)


//This is the diffuse colour that we'll set for each vertex in our cube.  The significant part
//is the 0x7F at the beginning, that's the alpha component and in this case it's set to 50%
//transparent.
const DWORD cube_colour=0x7FFFFFFF;

//The normal is a 3 component vector (x,y,z) which indicates the direction that the vertex
//faces.  For example, on the front face of our cube, the normal is set to (0,0,-1) which is
//pointing directly along the Z-axis towards the screen.  The back face is set to (0,0,1)
//which is pointing in the opposite direction.
my_vertex g_vertices[] ={
   {-1.0f,-1.0f,-1.0f,   0.0f, 0.0f,-1.0f,   cube_colour, 0.0f, 1.0f },  //Front face
   {-1.0f, 1.0f,-1.0f,   0.0f, 0.0f,-1.0f,   cube_colour, 0.0f, 0.0f },
   { 1.0f, 1.0f,-1.0f,   0.0f, 0.0f,-1.0f,   cube_colour, 1.0f, 0.0f },
   { 1.0f, 1.0f,-1.0f,   0.0f, 0.0f,-1.0f,   cube_colour, 1.0f, 0.0f },
   { 1.0f,-1.0f,-1.0f,   0.0f, 0.0f,-1.0f,   cube_colour, 1.0f, 1.0f },
   {-1.0f,-1.0f,-1.0f,   0.0f, 0.0f,-1.0f,   cube_colour, 0.0f, 1.0f },

   { 1.0f,-1.0f, 1.0f,   0.0f, 0.0f, 1.0f,   cube_colour, 0.0f, 1.0f },  //Back face
   { 1.0f, 1.0f, 1.0f,   0.0f, 0.0f, 1.0f,   cube_colour, 0.0f, 0.0f },
   {-1.0f, 1.0f, 1.0f,   0.0f, 0.0f, 1.0f,   cube_colour, 1.0f, 0.0f },
   {-1.0f, 1.0f, 1.0f,   0.0f, 0.0f, 1.0f,   cube_colour, 1.0f, 0.0f },
   {-1.0f,-1.0f, 1.0f,   0.0f, 0.0f, 1.0f,   cube_colour, 1.0f, 1.0f },
   { 1.0f,-1.0f, 1.0f,   0.0f, 0.0f, 1.0f,   cube_colour, 0.0f, 1.0f },

   {-1.0f, 1.0f,-1.0f,   0.0f, 1.0f, 0.0f,   cube_colour, 0.0f, 1.0f },  //Top face
   {-1.0f, 1.0f, 1.0f,   0.0f, 1.0f, 0.0f,   cube_colour, 0.0f, 0.0f },
   { 1.0f, 1.0f, 1.0f,   0.0f, 1.0f, 0.0f,   cube_colour, 1.0f, 0.0f },
   { 1.0f, 1.0f, 1.0f,   0.0f, 1.0f, 0.0f,   cube_colour, 1.0f, 0.0f },
   { 1.0f, 1.0f,-1.0f,   0.0f, 1.0f, 0.0f,   cube_colour, 1.0f, 1.0f },
   {-1.0f, 1.0f,-1.0f,   0.0f, 1.0f, 0.0f,   cube_colour, 0.0f, 1.0f },

   { 1.0f,-1.0f,-1.0f,   0.0f,-1.0f, 0.0f,   cube_colour, 0.0f, 1.0f },  //Bottom face
   { 1.0f,-1.0f, 1.0f,   0.0f,-1.0f, 0.0f,   cube_colour, 0.0f, 0.0f },
   {-1.0f,-1.0f, 1.0f,   0.0f,-1.0f, 0.0f,   cube_colour, 1.0f, 0.0f },
   {-1.0f,-1.0f, 1.0f,   0.0f,-1.0f, 0.0f,   cube_colour, 1.0f, 0.0f },
   {-1.0f,-1.0f,-1.0f,   0.0f,-1.0f, 0.0f,   cube_colour, 1.0f, 1.0f },
   { 1.0f,-1.0f,-1.0f,   0.0f,-1.0f, 0.0f,   cube_colour, 0.0f, 1.0f },

   {-1.0f,-1.0f, 1.0f,  -1.0f, 0.0f, 0.0f,   cube_colour, 0.0f, 1.0f },  //Left face
   {-1.0f, 1.0f, 1.0f,  -1.0f, 0.0f, 0.0f,   cube_colour, 0.0f, 0.0f },
   {-1.0f, 1.0f,-1.0f,  -1.0f, 0.0f, 0.0f,   cube_colour, 1.0f, 0.0f },
   {-1.0f, 1.0f,-1.0f,  -1.0f, 0.0f, 0.0f,   cube_colour, 1.0f, 0.0f },
   {-1.0f,-1.0f,-1.0f,  -1.0f, 0.0f, 0.0f,   cube_colour, 1.0f, 1.0f },
   {-1.0f,-1.0f, 1.0f,  -1.0f, 0.0f, 0.0f,   cube_colour, 0.0f, 1.0f },

   { 1.0f,-1.0f,-1.0f,   1.0f, 0.0f, 0.0f,   cube_colour, 0.0f, 1.0f },  //Right face
   { 1.0f, 1.0f,-1.0f,   1.0f, 0.0f, 0.0f,   cube_colour, 0.0f, 0.0f },
   { 1.0f, 1.0f, 1.0f,   1.0f, 0.0f, 0.0f,   cube_colour, 1.0f, 0.0f },
   { 1.0f, 1.0f, 1.0f,   1.0f, 0.0f, 0.0f,   cube_colour, 1.0f, 0.0f },
   { 1.0f,-1.0f, 1.0f,   1.0f, 0.0f, 0.0f,   cube_colour, 1.0f, 1.0f },
   { 1.0f,-1.0f,-1.0f,   1.0f, 0.0f, 0.0f,   cube_colour, 0.0f, 1.0f },

};

//This handy macro makes sure that we always reference the right number of
//vertices.  This way we always allocate a correctly sized vertex buffer.
#define NUM_VERTICES (sizeof(g_vertices)/sizeof(my_vertex))

//Note:We're using the dhPrimitive class to wrap our Vertex Buffers.  Since the code
//to allocate and otherwise mess with VBs is in all of the previous examples I thought
//it would be nice to trim down the code where possible to focus on the new material.
//The dhPrimitive class is available at http://www.drunkenhyena.com/docs/dhPrimitive.phtml

//For this we need 2 cubes.  The inner cube will have the normals pointing inwards, so
//that we see the back of our cube, and the outer cube has them facing outwards.  Otherwise
//the lighting would be wrong.
dhSimpleMesh outer_cube(NUM_VERTICES,D3D8T_CUSTOMVERTEX,D3DPT_TRIANGLELIST,NUM_VERTICES/3);
dhSimpleMesh inner_cube(NUM_VERTICES,D3D8T_CUSTOMVERTEX,D3DPT_TRIANGLELIST,NUM_VERTICES/3);

//Our rotation speed for the X-axis & Y-axis.  Initially 0.
float x_speed=0.0f;
float y_speed=0.0f;

//Initial position on the Z-axis
float z_pos=0.0f;

//What kind of texture filtering should we use?
DWORD filter=0;

//Track whether lighting is on or off.
BOOL lighting_on=TRUE;

//Our one lonely light
D3DLIGHT8 light;

// WinMain is the first function called by Windows when our app is run.  It's the entry
// point of our application.
int APIENTRY WinMain(HINSTANCE p_instance,HINSTANCE p_prev_instance,LPSTR p_cmd_line,int p_show){

   // Set our global instance handle so we don't have to pass it around
   g_instance=p_instance;

   //This function exists to quiet compiler warnings, see its definition for more detail
   NOP(p_prev_instance,p_cmd_line,p_show);

   // Prompt the user, Full Screen?  Windowed?  Cancel?
   ask_fullscreen();

   // Build our window.  Cover the screen if full-screen, otherwise make a standard window
   init_window();

   //Build the D3D objects we'll require
   init_d3d();

   //One-time preparation of objects and other stuff required for rendering
   init_scene();

   float booga=1e10;

   //Loop until the user aborts (closes the window or hits a key)
   while(!g_app_done){
      //Check for window messages
      message_pump();

      //Draw our incredibly cool graphics
      render();
   }

   kill_scene();

   //Clean up all of our Direct3D objects
   kill_d3d();

   //Close down our window
   kill_window();

   //Exit happily
   return 0;
}

// Procedure: NOP
// Whazzit:This procedure does nothing.  If set to a high warning level
//         (which I like to do) the compiler will complain because the
//         parameters passed into WinMain are never used.  The purpose
//         of this procedure is to make it think that they are used, so
//         it doesn't complain.  
void NOP(HINSTANCE p_prev_instance,LPSTR p_cmd_line,int p_show){

   p_prev_instance=p_prev_instance;
   p_cmd_line=p_cmd_line;
   p_show=p_show;
}


// Procedure: message_pump
// Whazzit:Checks the message queue to see if any windows messages
//         (window is closing, window needs repainting, etc)
//         are waiting and if there are, the messages are dispatched
//         to our message handler.
void message_pump(void){
MSG msg;

   if(PeekMessage(&msg, NULL, 0, 0,PM_REMOVE)){
      TranslateMessage(&msg);
      DispatchMessage(&msg);
   }

}


// Function:init_d3d
// Whazzit:Sets up Direct3D and creates the device.  The device is create differently
//         if we're full-screen as opposed to running in a desktop window.
void init_d3d(void){
HRESULT hr;
D3DPRESENT_PARAMETERS d3dpp;
D3DDISPLAYMODE display_mode;

   //Create Direct3D8, this is the first thing you have to do in any D3D8 program
   g_D3D = Direct3DCreate8( D3D_SDK_VERSION );
   if(!g_D3D ){
      FatalError("Error getting Direct3D");
   }

   //Get the current(desktop) display mode.  This is really only needed if
   //we're running in a window.
   hr=g_D3D->GetAdapterDisplayMode(D3DADAPTER_DEFAULT,&display_mode);
   if(FAILED(hr)){
      FatalError("Error getting display mode\n");
   }

   //Clear out our D3DPRESENT_PARAMETERS structure.  Even though we're going
   //to set virtually all of it members, it's good practice to zero it out first.
   ZeroMemory(&d3dpp,sizeof(d3dpp));

   //Whether we're full-screen or windowed these are the same.
   d3dpp.SwapEffect     = D3DSWAPEFFECT_DISCARD; // Throw away previous frames, we don't need them
   d3dpp.hDeviceWindow  = g_main_window;  //This is our main (and only) window
   d3dpp.BackBufferCount= 1;  //We only need a single back buffer

   //We need a Z-Buffer so everything will be drawn properly
   d3dpp.EnableAutoDepthStencil = TRUE;
   d3dpp.AutoDepthStencilFormat = D3DFMT_D16;

   // BackBufferWidth/Height have to be set for full-screen apps, these values are
   //used (along with BackBufferFormat) to determine the display mode.
   //They aren't needed in windowed mode since the size of the window will be used.
   // BackBufferFormat is the pixel format we want.  In windowed mode we use the same
   //format as the desktop (which we found by using GetAdapterDisplayMode() above).
   //In full-screen we need to find a pixel format we like, see find_16bit_mode()
   //below for more details.
   if(g_fullscreen){
      d3dpp.Windowed          = FALSE;
      d3dpp.BackBufferWidth   = g_width;
      d3dpp.BackBufferHeight  = g_height;
      d3dpp.BackBufferFormat  = find_16bit_mode();
   }else{
      d3dpp.Windowed          = TRUE;
      d3dpp.BackBufferFormat  = display_mode.Format;
   }


   hr=g_D3D->CreateDevice(D3DADAPTER_DEFAULT, //The default adapter, on a multimonitor system
                                              //there can be more than one.
                           //Use hardware acceleration rather than the software renderer
                          D3DDEVTYPE_HAL,
                          //Our Window
                          g_main_window,
                          //Process vertices in software. This is slower than in hardware,
                          //But will work on all graphics cards.
                          D3DCREATE_SOFTWARE_VERTEXPROCESSING,
                          //Our D3DPRESENT_PARAMETERS structure, so it knows what we want to build
                          &d3dpp,
                          //This will be set to point to the new device
                          &g_d3d_device);
   if(FAILED(hr)){
      FatalError("Error creating device\n");
   }


}
// Function:kill_d3d
// Whazzit:Releases all of our D3D resources in the opposite order from their creation.
//         Note-Since we initially set the pointers to be NULL, we can safely test them
//         for a non-NULL state and we know if they've been created.  Thus we never Release
//         something we didn't create (which causes bad things to happen).
void kill_d3d(void){

   if(g_d3d_device){
      g_d3d_device->Release();
      g_d3d_device=NULL;
   }

   if(g_D3D){
      g_D3D->Release();
      g_D3D=NULL;
   }

}
// Function:init_light
// Whazzit:Sets up the fields required in our light structure.
// Note: The light is statically allocated, so no kill_light() is required
void init_light(void){

   //Zero out our light structure and fill in the fields we need
   ZeroMemory( &light, sizeof(D3DLIGHT8) );

   //A point light shines in all directions and it's position is given in world coordinates.
   light.Type=D3DLIGHT_POINT;

   //Set what colour the light is.
   light.Diffuse.r=1.0f;
   light.Diffuse.g=1.0f;
   light.Diffuse.b=1.0f;

   //The light will not illuminate objects out of this range.  This value is a lot larger
   //than is necessary, but it doesn't hurt.
   light.Range=1000.0f;

   //This positions the light at the same location as our camera.
   light.Position=D3DXVECTOR3(0.0f,0.0f,-8.0f);

   //This controls how the light 'falls off' or has less intensity as the distance increases.
   //For now we set it so it's at full intensity for its full range (no fall off).
   light.Attenuation0=1.0f;

   //When dealing with lights in D3D you often refer to them by an light index.  This call
   //to SetLight associates our light with index 0.
   g_d3d_device->SetLight( 0, &light );

   //Enable our light.  Note this doesn't turn on lighting, it states that this light should be
   //used when lighting is on.
   g_d3d_device->LightEnable( 0, TRUE);

}
// Function:init_scene
// Whazzit:One-time preparation of objects required for rendering.
void init_scene(void){
HRESULT hr;
D3DXMATRIX view_matrix;
D3DXMATRIX matProj;

   init_light();

   //The following SetTextureStageState calls determine how textures will
   //be applied to your objects in all of the following Draw operations.

   //D3DTSS_COLOROP specifies how the colour of each pixel will be determined
   //In this case it's set to D3DTOP_MODULATE which means the colour is determined
   //by combining COLORARG1 & COLORARG2.
   g_d3d_device->SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_MODULATE);

   //D3DTSS_COLORARG1 is set to D3DTA_TEXTURE which means that colour1 is
   //entirely taken from the texture and nothing else.  D3DTSS_COLORARG2
   //is set to D3DTA_CURRENT which will be our vertex colour if lighting
   //is turned off, or the result of the lighting if it is on.
   g_d3d_device->SetTextureStageState(0,D3DTSS_COLORARG1,D3DTA_TEXTURE);
   g_d3d_device->SetTextureStageState(0,D3DTSS_COLORARG2,D3DTA_CURRENT);

   //Similarly to the COLOR TextureStageStates, the D3DTOP_SELECTARG1
   //says to take the result from arguement 1. We set D3DTSS_ALPHAARG1
   //to D3DTA_DIFFUSE, which makes the alpha value for each vertex come
   //completely from the diffuse colour of the vertex(which we set to 0x7f
   // or 50% transparent).
   g_d3d_device->SetTextureStageState(0,D3DTSS_ALPHAOP,D3DTOP_SELECTARG1);
   g_d3d_device->SetTextureStageState(0,D3DTSS_ALPHAARG1,D3DTA_DIFFUSE);

   //The next 2 RenderStates determine how the blending is done.
   //The source is our object, destination is the drawing surface.
   //SRCBLEND specifies that the final colour should be 50% (because
   //that's what we set our souce alpha to) from our source and 50%
   //(INVSRCALPHA, 100%-SRCALPHA(50% in this case))
   g_d3d_device->SetRenderState(D3DRS_SRCBLEND,D3DBLEND_SRCALPHA);
   g_d3d_device->SetRenderState(D3DRS_DESTBLEND,D3DBLEND_INVSRCALPHA);

   //This enables Alpha Blending
   g_d3d_device->SetRenderState(D3DRS_ALPHABLENDENABLE,TRUE);

   //The MAGFILTER specifies how the texture should be filtered when it's
   //drawn larger than it's real size, MINFILTER is for when it's drawn smaller
   //than it's real size.
   //Initially set it to NONE.
   g_d3d_device->SetTextureStageState(0,D3DTSS_MAGFILTER,D3DTEXF_NONE);
   g_d3d_device->SetTextureStageState(0,D3DTSS_MINFILTER,D3DTEXF_NONE);

   //Turn on D3D lighting.
   g_d3d_device->SetRenderState(D3DRS_LIGHTING,TRUE);

   //Ambient light is the default lighting level of the area.  Anything not hit
   //by our light will have this level of lighting.
   g_d3d_device->SetRenderState(D3DRS_AMBIENT,0x00888888);

   // Turn on the zbuffer.  The ZBuffer makes sure that objects appear in the
   //proper order.  Without it we'd have to make sure we draw our triangles
   //in back-to-front order, with a ZBuffer we don't have to worry about it.
   g_d3d_device->SetRenderState(D3DRS_ZENABLE,TRUE);

   //Given a pointer to our device and a file name for our bitmap,
   //this function loads our texture and set g_texture to point to it.
   hr=D3DXCreateTextureFromFile( g_d3d_device, "glass.bmp",&g_texture);
   if(FAILED(hr)){
      FatalError("Error loading texture");
   }

   //Build our outer_cube
   outer_cube.SetDevice(g_d3d_device);

   //Allocate space for the cube
   if(!outer_cube.Alloc()){
      FatalError("Error creating outer_cube vertex buffer");
   }

   //Copy vertices into the cube
   outer_cube.Fill(&g_vertices);


   //Change our normals to be their inverse (so they point the other way)
   int count;
   for(count=0;count < NUM_VERTICES;count++){

      g_vertices[count].nx*=-1.0f;
      g_vertices[count].ny*=-1.0f;
      g_vertices[count].nz*=-1.0f;

   }

   //Build our inner cube with our reversed normals
   inner_cube.SetDevice(g_d3d_device);

   if(!inner_cube.Alloc()){
      FatalError("Error creating inner_cube vertex buffer");
   }

   inner_cube.Fill(&g_vertices);

   //Here we build our View Matrix, think of it as our camera.
   //First we specify that our viewpoint is 8 units back on the Z-axis
   //We are looking towards the origin
   //And the y-axis is up
   D3DXMatrixLookAtLH(&view_matrix,&D3DXVECTOR3( 0.0f, 0.0f,-8.0f ),
                                   &D3DXVECTOR3( 0.0f, 0.0f, 0.0f ),
                                   &D3DXVECTOR3( 0.0f, 1.0f, 0.0f ));

   //Since our 'camera' will never move, we can set this once at the
   //beginning and never worry about it again
   g_d3d_device->SetTransform(D3DTS_VIEW,&view_matrix);


   D3DXMatrixPerspectiveFovLH(&matProj, //Result Matrix
                              D3DX_PI/4,//Field of View, in radians. (PI/4) is typical
                              ((float)g_width / (float)g_height),     //Aspect ratio
                              1.0f,     //Near view plane
                              100.0f ); // Far view plane

   //Our Projection matrix won't change either, so we set it now and never touch
   //it again.
   g_d3d_device->SetTransform( D3DTS_PROJECTION, &matProj );

}
// Function:kill_scene
// Whazzit:Clean up any objects we required for rendering.
void kill_scene(void){

   if(g_texture){
      g_texture->Release();
      g_texture=NULL;
   }

   outer_cube.Free();

   inner_cube.Free();

}
// Function:find_16bit_mode
// Whazzit:Tests a couple of 16-bit modes to see if they are supported.  Virtually every graphics
//         card in existance will support one of these 2 formats.
D3DFORMAT find_16bit_mode(void){
HRESULT hr;

   //First we test for R5G6B5.  All 16-bits are used in this format giving us a full 64K worth
   //worth of colours
   hr=g_D3D->CheckDeviceType(D3DADAPTER_DEFAULT,D3DDEVTYPE_HAL,D3DFMT_R5G6B5,D3DFMT_R5G6B5,FALSE);
   if(SUCCEEDED(hr)){
      OutputDebugString("D3DFMT_R5G6B5\n");
      return D3DFMT_R5G6B5;
   }

   //Next try X1R5G5B5. Since 1 bit is wasted it's technically a 15-bit mode and only
   //provides 32K colours, though you'd be hard pressed to tell the difference between
   //15- & 16-bit modes.
   hr=g_D3D->CheckDeviceType(D3DADAPTER_DEFAULT,D3DDEVTYPE_HAL,D3DFMT_X1R5G5B5,D3DFMT_X1R5G5B5,FALSE);
   if(SUCCEEDED(hr)){
      OutputDebugString("D3DFMT_X1R5G5B5\n");
      return D3DFMT_X1R5G5B5;
   }

   //This is a freaky card.  Complain and bail out.
   FatalError("Couldn't find a decent mode\n");

   //Won't actually hit this line since FatalError() kills us, but it makes the compiler happy.
   return (D3DFORMAT)NULL;
}
// Function:ask_fullscreen
// Whazzit:Ask the user if they would like to run in full-screen or windowed mode or if they
//         would like to Cancel (abort).
void ask_fullscreen(void){
int full_result;

   full_result=MessageBox(NULL,"Would you like to run in fullscreen mode?",g_app_name,
                          MB_YESNOCANCEL|MB_ICONQUESTION);
   switch(full_result){
      case IDCANCEL:
         OutputDebugString("User Abort");
         MessageBox(NULL,"User Abort",g_app_name,MB_OK);
         exit(5);
         break;
      case IDNO:
         g_fullscreen=false;
         break;
      case IDYES:
         g_fullscreen=true;
         break;
      case 0:
         OutputDebugString("Couldn't open MessageBox, dying");
         exit(10);
         break;
   }

}
// Function: render
// Whazzit:Clears the screen to a pseudo-random colour, draws a triangle and a square
//         and then presents the results.
void render(void){
D3DXMATRIX matWorld;
D3DXMATRIX rot_x_matrix;   //Our rotation matrix for the x-axis
D3DXMATRIX rot_y_matrix;   //Our rotation matrix for the y_axis
D3DXMATRIX rot_matrix;     //Final rotation matrix
D3DXMATRIX trans_matrix;   //Our translation matrix
static float rot_x=0;      //Tracks rotation for the x-axis
static float rot_y=0;      //Tracks rotation for the y-axis

   rot_x+=x_speed;
   rot_y+=y_speed;


   //Clear the buffer to our new colour and clear the ZBuffer
   g_d3d_device->Clear( 0, NULL, D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER , D3DCOLOR_XRGB(0,0,0), 1.0f, 0 );

   //Notify the device that we're ready to render
   if(SUCCEEDED(g_d3d_device->BeginScene())){

      //Vertex shaders are a complex topic, but you can do some amazing things with them
      //For this example we're not creating one, so we tell Direct3D that we're just
      //using a plain vertex format.
      g_d3d_device->SetVertexShader(D3D8T_CUSTOMVERTEX);

      //Here we set our texture to be the current texture in the first texture 'stage'
      //There can be up to 8 textures and they can be blended in a variety of ways.
      //For now we only need one.
      g_d3d_device->SetTexture( 0, g_texture );

      //Set up the rotation matrix for the cube
      D3DXMatrixRotationY(&rot_y_matrix,rot_y);
      D3DXMatrixRotationX(&rot_x_matrix,rot_x);

      //Combine the 2 matrices to get our final Rotation Matrix
      D3DXMatrixMultiply(&rot_matrix,&rot_x_matrix,&rot_y_matrix);

      //Translate out cube in/out of the screen
      D3DXMatrixTranslation(&trans_matrix,0.0f,0.0f,z_pos);

      //Build our final World Matrix
      D3DXMatrixMultiply(&matWorld,&rot_matrix,&trans_matrix);

      //Set our World Matrix
      g_d3d_device->SetTransform(D3DTS_WORLD,&matWorld );

      //We set our culling to target faces whose vertices are specified
      //in Clockwise order.  This is the reverse of the default.  This way
      //the far faces are kept, and near faces are culled.
      g_d3d_device->SetRenderState(D3DRS_CULLMODE,D3DCULL_CW);

      //Set D3D to read from our inner cube.  We draw the inner cube first.
      //The faces that our near the camera will be culled away, leaving the
      //only the faces pointed at us.
      //When doing alphablending it's important to draw objects that are far
      //away first, that way the front objects have something to blend with.
      g_d3d_device->SetStreamSource(0,inner_cube.GetVB(),sizeof(my_vertex));

      //Draw our cube
      g_d3d_device->DrawPrimitive(D3DPT_TRIANGLELIST,0,NUM_VERTICES/3);


      //Set D3D to read from our outer cube.  The back faces will be culled
      //away leaving the faces from our inner cube untouched.  Only the front
      //faces will be drawn and they'll blend nicely.
      g_d3d_device->SetStreamSource(0,outer_cube.GetVB(),sizeof(my_vertex));

      //Set culling to target Counter-Clockwise faces.  This is the default.
      //Now the near faces will be kept (since they're facing us) and the
      //far (facing away) faces will be culled.
      g_d3d_device->SetRenderState(D3DRS_CULLMODE,D3DCULL_CCW);

      //Draw our cube
      g_d3d_device->DrawPrimitive(D3DPT_TRIANGLELIST,0,NUM_VERTICES/3);

      //Notify the device that we're finished rendering for this frame
      g_d3d_device->EndScene();
   }

   //After drawing, empty the first texture stage.  When you SetTexture a texture
   //Direct3D adds to it's internal reference count, this prevents a texture from
   //being released when it's still being used.
   //When you're done drawing it's a good idea to set all of your texture stages to NULL
   //so your textures can be released when they should be.
   g_d3d_device->SetTexture(0,NULL);

   //Show the results
   g_d3d_device->Present( NULL, NULL, NULL, NULL );

}
// Function: init_window
// Whazzit:Registers a window class and then creates our window.
void init_window(void){
ULONG window_width, window_height;
WNDCLASS window_class;
DWORD style;

   //Fill in all the fields for the WNDCLASS structure.  Window classes
   //are a sort of template for window creation.  You could create many
   //windows using the same window class.
   window_class.style          = CS_OWNDC;
   window_class.cbClsExtra     = 0;
   window_class.cbWndExtra     = 0;
   window_class.hInstance      = g_instance;
   window_class.hIcon          = LoadIcon(NULL,IDI_APPLICATION);
   window_class.hCursor        = LoadCursor(NULL,IDC_ARROW);
   window_class.hbrBackground  = (HBRUSH)GetStockObject(BLACK_BRUSH);
   window_class.lpszMenuName   = NULL;
   window_class.lpszClassName  = "DH Class";
   //Here we provide our default window handler, all windows messages
   //will be sent to this function.
   window_class.lpfnWndProc    = default_window_proc;


   //Register the class with windows
   if(!RegisterClass(&window_class)){
      FatalError("Error registering window class");
   }

   //If we're running full screen, we cover the desktop with our window.
   //This isn't necessary, but it provides a smoother transition for the
   //user, especially when we're going to change screen modes.
   if(g_fullscreen){
      window_width=GetSystemMetrics(SM_CXSCREEN);
      window_height=GetSystemMetrics(SM_CYSCREEN);
      style=WS_POPUP;
   }else{
      //In windowed mode, we just make the window whatever size we need.
      window_width=g_width;
      window_height=g_height;
      style=WS_OVERLAPPED|WS_SYSMENU;
   }


   g_main_window=CreateWindow("DH Class",g_app_name,style,
                              0,0,window_width,window_height,NULL,NULL,
                              g_instance,NULL);

   if(!g_main_window){
      FatalError("Error opening window");
   }

   //The next 3 lines just make sure that our window is visible and has the
   //input focus.  It's not strictly necessary, but it doesn't hurt to be
   //thorough.
   ShowWindow(g_main_window,SW_SHOW);
   UpdateWindow(g_main_window);
   SetFocus(g_main_window);

}
// Function: kill_window
// Whazzit:Closes the window, clean up any waiting messages, and then unregister 
//         our window class.  Note - This is not the standard Win32 way of cleaning
//         up your window.  The standard way involves putting the clean-up code in
//         your window handler, I don't like that method.
void kill_window(void){

   //Test if our window is valid
   if(g_main_window){
      if(!DestroyWindow(g_main_window)){
         //We failed to destroy our window, this shouldn't ever happen
         OutputDebugString("   Failed to DestroyWindow\n");
         MessageBox(NULL,"Destroy Window Failed",g_app_name,MB_OK|MB_ICONERROR|MB_TOPMOST);
      }else{
         MSG msg;
         //Clean up any pending messages
         while(PeekMessage(&msg, NULL, 0, 0,PM_REMOVE)){
            DispatchMessage(&msg);
         }
      }
      //Set our window handle to NULL just to be safe
      g_main_window=NULL;
   }

   //Unregister our window, if we had opened multiple windows using this
   //class, we would have to close all of them before we unregistered the class.
   if(!UnregisterClass("DH Class",g_instance)){
      OutputDebugString("   Failed to Unregister Window\n");
      MessageBox(NULL,"Unregister Failed",g_app_name,MB_OK|MB_ICONERROR|MB_TOPMOST);
   }

}
// Function:FatalError
// Whazzit:Close down all resources, alert the user and quit
void FatalError(const char *error_msg){

   kill_scene();

   kill_d3d();

   kill_window();

	OutputDebugString( error_msg );
   OutputDebugString("\n");
	MessageBox(NULL, error_msg,g_app_name, MB_OK );

   exit(5);

}
// Function:default_window_proc
// Whazzit:All Windows messages get passed through this function.  We only handle
//         a tiny subset of the available messages, all unhandled messages get
//         passed through to DefWindowProc() which is part of the Win32 API.
LRESULT CALLBACK default_window_proc(HWND hwnd,UINT msg,WPARAM wparam,LPARAM lparam){
HDC hdc;
PAINTSTRUCT ps;
int virt_key = (int) wparam; //Only valid if msg==WM_KEYDOWN

   switch(msg){
      case WM_KEYDOWN:  // A key has been pressed
         switch(virt_key){
            case VK_UP:
               x_speed-=0.005f;
               break;
            case VK_DOWN:
               x_speed+=0.005f;
               break;
            case VK_LEFT:
               y_speed-=0.005f;
               break;
            case VK_RIGHT:
               y_speed+=0.005f;
               break;
            case VK_NEXT:
               z_pos+=0.03f;
               break;
            case VK_PRIOR:
               z_pos-=0.03f;
               break;
            case 'F':
               filter++;
               if(filter > 2)
                  filter=0;
               switch(filter){
                  case 0:
                     g_d3d_device->SetTextureStageState(0,D3DTSS_MAGFILTER, D3DTEXF_NONE);
                     g_d3d_device->SetTextureStageState(0,D3DTSS_MINFILTER, D3DTEXF_NONE);
                     break;
                  case 1:
                     g_d3d_device->SetTextureStageState(0,D3DTSS_MAGFILTER, D3DTEXF_POINT);
                     g_d3d_device->SetTextureStageState(0,D3DTSS_MINFILTER, D3DTEXF_POINT);
                     break;
                  case 2:
                     g_d3d_device->SetTextureStageState(0,D3DTSS_MAGFILTER, D3DTEXF_LINEAR);
                     g_d3d_device->SetTextureStageState(0,D3DTSS_MINFILTER, D3DTEXF_LINEAR);
                     break;
               }
               break;
            case 'L':
               lighting_on=!lighting_on;
               g_d3d_device->SetRenderState(D3DRS_LIGHTING,lighting_on);
               break;
            case VK_ESCAPE:
               g_app_done=true;
               break;
         }
         return 0;
      case WM_PAINT:    //Repaint our window
         hdc=BeginPaint(hwnd,&ps);
         EndPaint(hwnd,&ps);
         return 0;
      case WM_CLOSE:    //User hit the Close Window button, end the app
         g_app_done=true;
         return 0;
      case WM_DESTROY:  //This window is being destroyed, tell Windows we're quitting
         PostQuitMessage(0);
         return 0;

   }

   return (DefWindowProc(hwnd,msg,wparam,lparam));

}
