
#define WIN32_LEAN_AND_MEAN		// Exclude rarely-used stuff from Windows headers
#include <D3DX8.h>
#include <stdio.h>
#include <mmsystem.h>

#include "dhEngine.h"
#include "dhSimpleMesh.h"
#include "dhFPSTimer.h"

//We use 2*PI a fair bit, so we predeclare it.
const float pi2=D3DX_PI * 2.0f;

//The following define compiles in a 'dhEngine-aware' version of dhFastFont
#define DHFF_USE_DHENGINE
#include "dhFastFont.h"

#pragma comment(lib,"d3d8.lib")
#pragma comment(lib,"winmm.lib")

#ifdef _DEBUG
   #pragma comment(lib,"dhEngineD.lib")
   #pragma comment(lib,"d3dx8d.lib")
#else
   #pragma comment(lib,"dhEngine.lib")
   #pragma comment(lib,"d3dx8.lib")
#endif

LRESULT CALLBACK default_window_proc(HWND hwnd,UINT msg,WPARAM wparam,LPARAM lparam);
void init_scene(void);
void kill_scene(void);
void Render(void);
void Update(void);
void load_object(dhSimpleMesh *p_mesh,const char *p_file_name);
void init_player(void);
void init_keys(void);

dhEngine g_engine;
bool g_app_done=false;

const char g_app_name[]="NeHe Style D3D8 Lesson 10";

const int g_width=800;  //Display width
const int g_height=600; //Display height

//Our display font.  Used to show an FPS counter
dhFastFont *g_font;

//Our timer to track our FPS
dhFPSTimer g_timer(g_app_name);

//All we need for our vertices is a position and texture coordinates
struct my_vertex{
   float x, y, z;
   float tu,tv;
};

//These index into our key array for movement, 'cuz Laura wouldn't let me use
//direct Keyboard querying
enum{
   KEY_UP=0,
   KEY_DOWN,
   KEY_LEFT,
   KEY_RIGHT,
   KEY_SPACE,
   KEY_COUNT
};
//Are the keys up or down?
bool g_key_state[KEY_COUNT];

my_vertex *g_vertices;

#define D3D8T_CUSTOMVERTEX (D3DFVF_XYZ|D3DFVF_TEX1)

//This defines our 'player'.
struct player_struct{

   D3DXVECTOR3 position;   //Current position
   D3DXVECTOR3 up;         //Up vector
   float angle;            //Rotation on the Y-axis (in radians)

} g_player;

//Our world is made up of 3 objects, a floor, a ceiling and set of walls
dhSimpleMesh g_floor;
dhSimpleMesh g_ceiling;
dhSimpleMesh g_walls;

//This is used to time our frames so we can scale movement appropriately.  We've
//done this in numerous tutorials already.
DWORD g_last_frame_time=0;


//Draw in filled mode or wireframe?  I added this to make debugging easier
bool g_fill_on=true;

int APIENTRY WinMain(HINSTANCE p_instance,HINSTANCE p_prev_instance,LPSTR p_cmd_line,int p_show){

   //Basic intialization of our engine
   g_engine.SetEngineAppName(g_app_name);
   g_engine.SetEngineInstance(p_instance);

   g_engine.SetEngineWidth(g_width);
   g_engine.SetEngineHeight(g_height);

   g_engine.SetEngineClearColour(0x00000000);
   g_engine.SetEngineClearZ(1.0f);

   g_engine.SetEngineWindowProc(default_window_proc);

   g_engine.SetEngineZBuffer(true);

   g_engine.SetEngineFullScreen(g_engine.AskFullScreen());


   g_engine.InitWindow();

   g_engine.InitD3D();

   init_keys();

   //Initialize our players position and heading
   init_player();

   init_scene();

   //Setup our font to display our FPS stats sin()
   g_font=new dhFastFont(&g_engine,"Data/font.png","Data/font.dat",D3DFMT_A8R8G8B8);
   if(!g_font->IsOkay()){
      g_engine.FatalError("Error loading font");
   }

   //Setup and then start our timer
   g_timer.SetDevice(g_engine.GetEngineDevice());

   g_timer.StartTimer();

   //Set our frame timer to a reasonable start value
   g_last_frame_time=timeGetTime();

   while(!g_app_done){

      g_engine.MessagePump();

      Update();

      Render();

      g_timer.Present();
   }

   //Log our timer stats
   g_timer.LogTimer();

   //Clean up our font
   delete g_font;

   kill_scene();

   g_engine.KillD3D();

   g_engine.KillWindow();


   return 0;
}
// Procedure:init_keys
// Whazzit:Initializes all of our keys to false (up)
void init_keys(void){
int count;

   for(count=0;count < KEY_COUNT;count++){
      g_key_state[count]=false;
   }

}

// Procedure:init_player
// Whazzit:initialize all data related to our 'player'
void init_player(void){

   //Our player/camera starts at (0,0) (x,z) and 1 unit off the ground to give him some
   //height
   g_player.position=D3DXVECTOR3(0.0f, 1.0f, 0.0f);

   //Our rotation angle on the Y-axis, we use this to calculate our LookAt point
   //in our View Matrix
   g_player.angle=0.0f;

   //Which way is up?
   g_player.up=D3DXVECTOR3(0.0f, 1.0f, 0.0f);
}

// Procedure:Update
// Whazzit:Calculates how long has past since our last frame.  Then updates our
//         player position and facing based on the time scale.
void Update(void){
DWORD current_time;
float time_scale;
D3DXMATRIX trans_matrix,rot_matrix,final_matrix;
D3DXMATRIX view_matrix;
float velocity,turn_rate;
D3DXVECTOR3 look_at;
bool changed_camera=false;

   //Here we calculate our time_scale which is the time since the previous
   //frame modified into a format we want (in this case we're dividing the
   //difference in milliseconds by 1000, which gives us a very small floating point
   //number.
   current_time=timeGetTime();

   time_scale=(current_time-g_last_frame_time)/1000.0f;

   g_last_frame_time=current_time;

   //Our velocity is based on our time scale so we have consistent movement
   //speed from computer to computer.  The value of (4 * time_scale) was derived
   //by rapid protoyping (I guessed values until I found one I liked).  The turn_rate
   //is similarly calculated.
   velocity=4.0f * time_scale;
   turn_rate=1.0f * time_scale;
   
   //Here we test for a key press that means we rotate(turn).  We use our
   //time_scale as the angle so that turning speed is consistant on any
   //speed of computer.  If we've turned we set the changed_angle flag
   //so we know to recalculate our view later in the function.
   if(g_key_state[KEY_RIGHT]){
      g_player.angle+=turn_rate;
      if(g_player.angle > pi2){
         g_player.angle-=pi2;
      }
      changed_camera=true;
   }else if(g_key_state[KEY_LEFT]){
      g_player.angle-=turn_rate;
      if(g_player.angle < 0){
         g_player.angle+=pi2;
      }
      changed_camera=true;
   }

   //Similar to the above code that test for turns, here we test for movement keys.
   //If we detect movement, we take the angle we're facing and use it with our velocity
   //to calculate our new position
   if(g_key_state[KEY_UP]){
      g_player.position.z+=(velocity) * (cosf(g_player.angle));
      g_player.position.x+=(velocity) * (sinf(g_player.angle));
      changed_camera=true;
   }else if(g_key_state[KEY_DOWN]){
      g_player.position.z-=(velocity) * (cosf(g_player.angle));
      g_player.position.x-=(velocity) * (sinf(g_player.angle));
      changed_camera=true;
   }

   //Since there is no collision detection, you can go through the walls
   //and wander forever, hitting space will bring you back to the room.
   if(g_key_state[KEY_SPACE]){  //Reset our position
      g_player.position.z=0.0f;
      g_player.position.x=0.0f;
      g_player.angle=0.0f;
      changed_camera=true;
   }

   //Only rebuild the View matrix if we've move or turned
   //There is some cost in rebuilding the view, D3D has to make some
   //internal adjustments, so we only do this when needed.   

   if(changed_camera ){
      look_at.x=sinf(g_player.angle)+g_player.position.x;
      look_at.y=g_player.position.y;
      look_at.z=cosf(g_player.angle)+g_player.position.z;


      D3DXMatrixLookAtLH(&view_matrix,
                         &g_player.position,
                         &look_at,
                         &g_player.up);

      g_engine.SetTransform(D3DTS_VIEW,&view_matrix);
   }

}
// Procedure:set_geometry_states
// Whazzit:This procedure sets the states we need to draw our world.
//         Note:The dhEngine filters out redundant calls.
void set_geometry_states(){

   g_engine.SetTextureStageState(0,D3DTSS_COLOROP,D3DTOP_SELECTARG1);
   g_engine.SetTextureStageState(0,D3DTSS_COLORARG1,D3DTA_TEXTURE);

   //We turn off culling so that we can see the room from the 'back'
   //There is no collision detection, so we can go thru the walls.
   g_engine.SetRenderState(D3DRS_CULLMODE,D3DCULL_NONE);
   
   g_engine.SetRenderState(D3DRS_ALPHABLENDENABLE,FALSE);
   g_engine.SetRenderState(D3DRS_LIGHTING,FALSE);

   g_engine.SetTextureStageState(0,D3DTSS_MAGFILTER, D3DTEXF_LINEAR);
   g_engine.SetTextureStageState(0,D3DTSS_MINFILTER, D3DTEXF_LINEAR);
   g_engine.SetTextureStageState(0,D3DTSS_MIPFILTER, D3DTEXF_LINEAR);

}

// Procedure:Render
// Whazzit:Draws our world
void Render(void){
char buffer[20];
float fps;

   //Clear the Colour & Z Buffers
   g_engine.ClearCZ();

   //Prepare our FPS string, this is used to show the FPS counter.
   fps=g_timer.GetFPS();
   sprintf(buffer,"FPS:%.1f",fps);


   if(SUCCEEDED(g_engine.BeginScene())){

      //Reset the states we need.  The dhEngine filters out redundant calls
      //so there's no real cost to this.
      set_geometry_states();

      //Draw our geometry, we keep this simple by using the dhSimpleMesh DrawWithContext
      //call which is not the most efficient thing to do here, but we've drawn geometry
      //in many other tutorials, so we want to focus on other things in this one.
      g_floor.DrawWithContext();

      g_ceiling.DrawWithContext();

      g_walls.DrawWithContext();

      //Draw our FPS counter
      g_font->Draw(buffer,0,0,0xFFFFFFFF);

   }
   g_engine.EndScene();

}
// Procedure:init_scene
// Whazzit:This is our standard scene setup, nothing new here
void init_scene(void){
D3DXMATRIX matrix;
float aspect_ratio;
D3DXVECTOR3 LookAt(0.0f,1.0f,1.0f);

   aspect_ratio=(float)g_width/(float)g_height;

   D3DXMatrixPerspectiveFovLH(&matrix, //Result Matrix
                              D3DX_PI/4,//Field of View, in radians. (PI/4) is typical
                              aspect_ratio,     //Aspect ratio
                              0.1f,     //Near view plane
                              50.0f ); // Far view plane

   g_engine.SetTransform(D3DTS_PROJECTION,&matrix);

   //We take our position & facing from the "player"
   D3DXMatrixLookAtLH(&matrix,
                      &g_player.position,
                      &LookAt,
                      &g_player.up);

   g_engine.SetTransform(D3DTS_VIEW,&matrix);

   //Load our world geometry
   load_object(&g_floor,"data/floor.txt");

   load_object(&g_ceiling,"data/ceiling.txt");

   load_object(&g_walls,"data/walls.txt");
}
// Procedure:kill_scene
// Whazzit: Cleans up our scene-specific data
void kill_scene(void){

   //Free all of our geometry
   g_floor.Free();

   g_ceiling.Free();

   g_walls.Free();

}
// Procedure:load_data
// Whazzit:Here we load our custom object format.  It's not an exciting format, but it stores
//         vertex info that we need as well as a texture name.
void load_data(dhSimpleMesh *p_primitive,FILE *p_file){
bool okay;
my_vertex *vert_ptr;
char buffer[256];
float x,y,z,tu,tv;
DWORD vert_count=0;
DWORD num_polys;

   //How many polys are there supposed to be?
   num_polys=p_primitive->GetPolyCount();

   vert_ptr=(struct my_vertex *)p_primitive->Lock();

   //Load data until we run out or an error occurs
   okay=true;
   while(okay){
      fgets(buffer,255,p_file);
      if(feof(p_file)){
         okay=false;
      }else if(strlen(buffer) >1  && buffer[0] !='#'){
         sscanf(buffer,"%f %f %f %f %f",&x,&y,&z,&tu,&tv);
         vert_ptr->x=x;
         vert_ptr->y=y;
         vert_ptr->z=z;
         vert_ptr->tu=tu;
         vert_ptr->tv=tv;
         vert_ptr++;
         vert_count++;
         if(vert_count > (num_polys*3)){
            g_engine.FatalError("Data Error!\n");
         }
      }

   }
   p_primitive->Unlock();   
}
// Procedure:load_object
// Whazzit:This function allocates our memory, fills it with our geometry data and
//         loads our textures.
void load_object(dhSimpleMesh *p_mesh,const char *p_file_name){
FILE *object_file;
char buffer[256];
HRESULT hr;
IDirect3DTexture8 *texture;
D3DXMATRIX matrix;
DWORD num_polys;


   //open our data file
   object_file=fopen(p_file_name,"r");
   if(!object_file){
      g_engine.FatalError("Failure to open object file");
   }

   //Get the texture name
   fgets(buffer,255,object_file);
   buffer[strlen(buffer)-1]='\0';

   //Load and assign texture
   hr=g_engine.CreateTextureFromFile(buffer,&texture);
   if(FAILED(hr)){
      g_engine.FatalError("Error loading texture");
   }
   p_mesh->SetTexture(texture);
   
   //Our primitive class will track the texture and free it when it's done
   //so we can release our hold on it.
   texture->Release();

   //Read the polygon count from the file
   fscanf(object_file,"NUM_POLYS %ld\n",&num_polys);

   //Set up our primitive class
   p_mesh->SetDevice(g_engine.GetEngineDevice());
   p_mesh->SetPrimitiveType(D3DPT_TRIANGLELIST);
   p_mesh->SetVertexSpec(D3D8T_CUSTOMVERTEX);

   p_mesh->SetPolyCount(num_polys);
   p_mesh->SetVertexCount(num_polys*3);

   D3DXMatrixIdentity(&matrix);
   p_mesh->SetWorldMatrix(&matrix);


   if(!p_mesh->Alloc()){
      g_engine.FatalError("Error allocating object mesh\n");
   }

   load_data(p_mesh,object_file);

}
// Procedure:default_window_proc
// Whazzit:This is our standard window message handler.  All of our keyboard handling
//         is done here.  Standard arrow key movement and SPACE to recenter our position.
//         We also allowing toggling between wireframe & solid fill modes.  It can be
//         very handy for debugging.
LRESULT CALLBACK default_window_proc(HWND hwnd,UINT msg,WPARAM wparam,LPARAM lparam){
char virt_key = (char)wparam;

   switch(msg){
      case WM_KEYDOWN:
         switch(virt_key){
            case VK_ESCAPE:
               g_app_done=true;
               break;
            case 'F':  //Toggle Fill Mode
               g_fill_on=!g_fill_on;
               if(g_fill_on){
                  g_engine.SetRenderState(D3DRS_FILLMODE ,D3DFILL_SOLID);
               }else{
                  g_engine.SetRenderState(D3DRS_FILLMODE ,D3DFILL_WIREFRAME);
               }
               break;
            case VK_UP:
               g_key_state[KEY_UP]=true;
               break;
            case VK_DOWN:
               g_key_state[KEY_DOWN]=true;
               break;
            case VK_LEFT:
               g_key_state[KEY_LEFT]=true;
               break;
            case VK_RIGHT:
               g_key_state[KEY_RIGHT]=true;
               break;
            case VK_SPACE:
               g_key_state[KEY_SPACE]=true;
               break;
         }
         return 0;
      case WM_KEYUP:
         switch(virt_key){
            case VK_UP:
               g_key_state[KEY_UP]=false;
               break;
            case VK_DOWN:
               g_key_state[KEY_DOWN]=false;
               break;
            case VK_LEFT:
               g_key_state[KEY_LEFT]=false;
               break;
            case VK_RIGHT:
               g_key_state[KEY_RIGHT]=false;
               break;
            case VK_SPACE:
               g_key_state[KEY_SPACE]=false;
               break;

         }
         return 0;
      case WM_LBUTTONDOWN: //user hit the left mouse button
      case WM_CLOSE:       //User hit the Close Window button, end the app
         g_app_done=true;
         return 0;
      case WM_DESTROY:  //This window is being destroyed, tell Windows we're quitting
         PostQuitMessage(0);
         return 0;
   }

   return (DefWindowProc(hwnd,msg,wparam,lparam));

}
