#include "libjsmidi.h"




static inline int ioread(jsmio* io, void* buf, size_t amt) {
    return (io->read)(io->fil,buf,amt);
}
static inline int ioreadc(jsmio* io) {
    unsigned char c;
    int r = ioread(io,&c,1);
    if (r < 1) return -1;
    return c;
}
static inline int ioreadf16(jsmio* io, void* buf) {
    char xbuf[2];
    int read=ioread(io,xbuf,2);
    ((char*)buf)[0]=xbuf[1];
    ((char*)buf)[1]=xbuf[0];
    return read;
}
static inline int ioreadf32(jsmio* io, void* buf) {
    char xbuf[4];
    int read=ioread(io,xbuf,4);
    ((char*)buf)[0]=xbuf[3];
    ((char*)buf)[1]=xbuf[2];
    ((char*)buf)[2]=xbuf[1];
    ((char*)buf)[3]=xbuf[0];
    return read;
}
static inline int iowrite(jsmio* io, void* buf, size_t amt) {
    return (io->write)(io->fil,buf,amt);
}
static inline int iowritef32(jsmio* io, void* buf) {
    char xbuf[4];
    xbuf[0]=((char*)buf)[3];
    xbuf[1]=((char*)buf)[2];
    xbuf[2]=((char*)buf)[1];
    xbuf[3]=((char*)buf)[0];
    return iowrite(io,xbuf,4);
}
static inline int iowritef16(jsmio* io, void* buf) {
    char xbuf[2];
    xbuf[0]=((char*)buf)[1];
    xbuf[1]=((char*)buf)[0];
    return iowrite(io,xbuf,2);
}
static inline int ioseek(jsmio* io, long offset, int whence) {
    return (io->seek)(io->fil,offset,whence);
}
static inline void ioclose(jsmio* io) {
    (io->close)(io->fil);
    free(io);
}
// CHANGE THIS IF YOUR MACHINE IS BIG ENDIAN!!!!
//#define LITTLE_ENDIAN

#ifdef LITTLE_ENDIAN
#define ioreadle16(io,buf) ioread(io,buf,2);
#define ioreadle32(io,buf) ioread(io,buf,4);
#define ioreadbe16(io,buf) ioreadf16(io,buf);
#define ioreadbe32(io,buf) ioreadf32(io,buf);

#define iowritele16(io,buf) iowrite(io,buf,2);
#define iowritele32(io,buf) iowrite(io,buf,4);
#define iowritebe16(io,buf) iowritef16(io,buf);
#define iowritebe32(io,buf) iowritef32(io,buf);

#else

#define ioreadbe16(io,buf) ioread(io,buf,2);
#define ioreadbe32(io,buf) ioread(io,buf,4);
#define ioreadle16(io,buf) ioreadf16(io,buf);
#define ioreadle32(io,buf) ioreadf32(io,buf);

#define iowritebe16(io,buf) iowrite(io,buf,2);
#define iowritebe32(io,buf) iowrite(io,buf,4);
#define iowritele16(io,buf) iowritef16(io,buf);
#define iowritele32(io,buf) iowritef32(io,buf);
#endif
/*typedef struct {
    jsmreadfunc read;
    jsmwritefunc write;
    jsmseekfunc seek;
    jsmclosefunc close;
    void* fil;
} jsmio;

typedef struct {
    ui32 miditime;
    ui32 millistime;
    ui8 track;
    ui8 status;
    ui16 len;
    union {
	ui8 sdata[3]; // for short messages
	ui8* pdata; // for sysex, meta
    };
} jsmevent;

typedef struct {
    ui32 numevents;
    jsmevent* events;
} midifile;
*/

#define SET_FUNCS(io,suffix)			\
    io->read=jsmread ## suffix;			\
    io->write=jsmwrite ## suffix;		\
    io->seek=jsmseek ## suffix;			\
    io->close=jsmclose ## suffix;

static int jsmreadfile(void* fil, void* buf, size_t amt) {
    return fread(buf,amt,1,(FILE*)fil);
}
static int jsmwritefile(void* fil, void* buf, size_t amt) {
    return fwrite(buf,1,amt,(FILE*)fil);
}
static int jsmseekfile(void* fil, long offset, int whence) {
    fseek((FILE*)fil,offset,whence);
    return ftell((FILE*)fil);
}
static void jsmclosefile(void* fil) {
    fclose((FILE*)fil);
}

jsmio* jsm_io_from_file(char* fname, char* mode) {
    FILE* fp = fopen(fname,mode);
    if (!fp) return NULL;
    jsmio* io = (jsmio*)malloc(sizeof(jsmio));
    io->fil=fp;
    SET_FUNCS(io,file);
    return io;
}

typedef struct {
    int mode;
    ui32 pos;
} iomem;
typedef struct {
    int mode;
    ui32 pos;
    ui32 length;
    ui8* buf;
} iomem_read;

typedef struct {
    int mode;
    ui32 pos;
    ui32 capacity;
    ui32* length;
    ui8** buf;
} iomem_write;

static int jsmreadmem(void* fil, void* buf, size_t amt) {
    iomem_read* mem=(iomem_read*)fil;
    if (mem->mode==1) return 0;
    if (mem->pos+amt>mem->length) amt=mem->length-mem->pos;
    if (mem->pos>=mem->length) return -1;
    if (amt>0) {
	memcpy(buf,(mem->buf)+mem->pos,amt);
	mem->pos += amt;
    }
    return amt;
}
static int jsmwritemem(void* fil, void* buf, size_t amt) {
    iomem_write* mem=(iomem_write*)fil;
    if (mem->mode==0) return 0;
    if (mem->pos+amt>mem->capacity) { // $#!+, buffer overflow 
	mem->capacity=(mem->pos+amt+127)&(~127);
	*mem->buf=realloc(*mem->buf,mem->capacity);
    }
    if (amt>0) {
	memcpy((*mem->buf)+mem->pos,buf,amt);
	mem->pos += amt;
    }
    if (mem->pos>*mem->length) *mem->length=mem->pos;
    return amt;
}
static int jsmseekmem(void* fil, long offset, int whence) {
    iomem* mem=(iomem*)fil;
    int len;
    switch (whence) {
    case SEEK_SET:
	mem->pos=whence;
	break;
    case SEEK_CUR:
	mem->pos+=whence;
	break;
    case SEEK_END:
	if (mem->mode==0) len=((iomem_read*)mem)->length;
	else len=*(((iomem_write*)mem)->length);
	mem->pos=len+whence;
	break;
    }
    return mem->pos;
}
static void jsmclosemem(void* fil) {
    iomem* mem=(iomem*)fil;
    if (mem->mode==1)  {
	iomem_write* memw=(iomem_write*)fil;
	*memw->buf = realloc(*memw->buf,*memw->length);
    }
	
    free(mem);
}

jsmio* jsm_io_read_mem(ui8* buf, int length) {
    iomem_read* iom = (iomem_read*)malloc(sizeof(iomem_read));
    jsmio* io = (jsmio*)malloc(sizeof(jsmio));
    iom->buf=buf;
    iom->length=length;
    iom->mode=0; // read
    iom->pos=0;
    io->fil=iom;
    SET_FUNCS(io,mem);
    return io;
}

jsmio* jsm_io_write_mem(ui8** buf, ui32* length, int icapacity) {
    iomem_write* iom = (iomem_write*)malloc(sizeof(iomem_read));
    jsmio* io = (jsmio*)malloc(sizeof(jsmio));
    iom->buf=buf;
    iom->length=length;
    iom->capacity=icapacity;
    iom->mode=1; // write
    iom->pos=0;
    io->fil=iom;
    SET_FUNCS(io,mem);
    return io;
}

int read_var(jsmio* io) {
    int accum=0;
    int rd=0;
    do {
	rd=ioreadc(io);
	if (rd==-1) return -1;
	accum <<= 7;
	accum += (rd&127);
    } while (rd&128);
    return accum;
	
}
void write_var(jsmio*io, int var) {
    char buf[6];
    int pos=5;
    int c;
    if (!var) {
	c=0; iowrite(io,&c,1);
    }
    while (var) {
	buf[pos]=(var&127);
	var >>=7;
	buf[pos] |= 128;
	pos--;
    }
    buf[5] &= 127;
    iowrite(io,buf+pos+1,5-pos);
}

#define MIDI_HEADER "MThd"
#define MIDI_THEADER "MTrk"

#define ERROR(string) do { return NULL; } while (0)

#define NO_MORE_EVENTS (~0)

static void readevent(jsmio* io, jsmevent* evt, ui32* ltime, ui8* lstatus, int track) {
    int dtime = read_var(io);
    if (dtime==-1) {
	*ltime=NO_MORE_EVENTS;
	return;
    }
    evt->status=*lstatus;
    int firstbt = ioreadc(io);
    int sbyte;
    if (firstbt>=128) {
	evt->status=firstbt;
    } else {
	if (*lstatus==0xFF) fprintf(stderr,"WARNING: meta event with running status!\n");
    }
    *lstatus=evt->status;
    *ltime += dtime;
    evt->miditime = *ltime;
    evt->track=track;
    switch (evt->status&0xF0) {
    case 0x80:
    case 0x90:
    case 0xA0:
    case 0xB0:
    case 0xE0:
	if (firstbt>=128) firstbt = ioreadc(io);
	sbyte = ioreadc(io);
	evt->len=2;
	evt->sdata[0]=firstbt;	
	evt->sdata[1]=sbyte;
	break;
    case 0xC0:
    case 0xD0:
	if (firstbt>=128) firstbt = ioreadc(io);
	evt->sdata[0]=firstbt;	
	evt->len=1;
	break;
    case 0xF0: // special
	if (evt->status==0xFF) {
	    if (firstbt>=128) { // very lenient, this...
		evt->metanum=ioreadc(io);
	    } else {
		evt->metanum=firstbt;
	    }
	    evt->len=read_var(io);
	    evt->pdata=malloc(evt->len+1);
	    ioread(io,evt->pdata,evt->len);
	    evt->pdata[evt->len]=0; // so it will work as a string
	} else if (evt->status==0xF0 || evt->status==0xF7) {
	    evt->len=read_var(io);
	    evt->pdata=malloc(evt->len);
	    ioread(io,evt->pdata,evt->len);
	    
	} else {
	    evt->len=0;
	}
    }
}
static void writeevent(jsmio* io, jsmevent* evt, ui32* ltime, ui8* lstatus) {
    int dtime = evt->miditime-*ltime;
    *ltime=evt->miditime;
    write_var(io,dtime);
    if (*lstatus!=evt->status) {
	iowrite(io,&evt->status,1);
    }
    *lstatus=evt->status;
    if ((evt->status&0xF0)==0xF0) {
	*lstatus=0; // force next status to be written
	if (evt->status==0xFF) {
	    iowrite(io,&evt->metanum,1);
	}
	write_var(io,evt->len);
	iowrite(io,evt->pdata,evt->len);
	return;
    }
    iowrite(io,evt->sdata,evt->len);
}

void jsm_makenoteon(jsmevent* evt, ui32 time, int track, int channel, int note, int vol) {
    evt->miditime=time;
    evt->track=track;
    evt->status=0x90+channel;
    evt->len=2;
    evt->sdata[0]=note;
    evt->sdata[1]=vol;
}
void jsm_makenoteoff(jsmevent* evt, ui32 time, int track, int channel, int note) {
    evt->miditime=time;
    evt->track=track;
    evt->status=0x90+channel;
    evt->len=2;
    evt->sdata[0]=note;
    evt->sdata[1]=0;
}
void jsm_makepitchbend(jsmevent* evt, ui32 time, int track, int channel, int bend) {
    evt->miditime=time;
    evt->track=track;
    evt->status=0xE0+channel;
    evt->len=2;
    evt->sdata[0]=(bend>>7)&127;
    evt->sdata[1]=bend&127;
}
void jsm_makeprogramchange(jsmevent* evt, ui32 time, int track, int channel, int inst) {
    evt->miditime=time;
    evt->track=track;
    evt->status=0xC0+channel;
    evt->len=1;
    evt->sdata[0]=inst;
}
void jsm_makecontrolchange(jsmevent* evt, ui32 time, int track, int channel, int num, int val) {
    evt->miditime=time;
    evt->track=track;
    evt->status=0xB0+channel;
    evt->len=2;
    evt->sdata[0]=num;
    evt->sdata[1]=val;
}
void jsm_makemeta(jsmevent* evt, ui32 time, int track, int num, const void* buf, int len) {
    evt->miditime=time;
    evt->track=track;
    evt->status=0xFF;
    evt->metanum=num;
    evt->len=len;
    evt->pdata=malloc(len);
    memcpy(evt->pdata,buf,len);
}
void jsm_makemetanum(jsmevent* evt, ui32 time, int track, int num, int val, int len) {
    ui8* buf=malloc(len);
    int i;
    for (i=0;i<len;i++) {
	buf[i]=(val>>((len-i-1)*8))&255;
    }
    jsm_makemeta(evt,time,track,num,buf,len);
    free(buf);
}
void jsm_makemetastr(jsmevent* evt, ui32 time, int track, int num, const char* buf) {
    jsm_makemeta(evt,time,track,num,buf,strlen(buf));
}

void jsm_addevent(midifile* mf, jsmevent* evt) {
    if (mf->numevents+1 >= mf->evtcap) {
	mf->evtcap = mf->numevents+16;
	mf->events=realloc(mf->events,mf->evtcap*sizeof(jsmevent));
    }
    mf->events[mf->numevents++]=*evt;
    
}
void jsm_freemidi(midifile* mf) {
    int i;
    for (i=0;i<mf->numevents;i++) {
	jsmevent* cur = &mf->events[i];
	switch(cur->status) {
	case 0xF0:
	case 0xF7:
	case 0xFF:
	    free(cur->pdata);
	}
    }
    free(mf->events);
    free(mf);
}
midifile* jsm_newmidi(int numtracks, int divis, int initcap) {
    midifile* mf = malloc(sizeof(midifile));
    mf->numtracks=numtracks; 
    mf->divis=divis;
    mf->evtcap=initcap;
    mf->numevents=0;
    mf->events = (jsmevent*)malloc(mf->evtcap*sizeof(jsmevent));
    return mf;
}
void jsm_setcapacity(midifile* mf, int newcap) {
    mf->evtcap=newcap;
    mf->events = (jsmevent*)realloc(mf->events,mf->evtcap*sizeof(jsmevent));
}
midifile* jsm_readmidi(jsmio* io, int autoclose) {
    char tbuf[5];
    ioread(io,tbuf,4);
    tbuf[4]=0;
    if (strcmp(tbuf,MIDI_HEADER)) ERROR("Not a midi file");
    int len;
    ioreadbe32(io,&len);
    if (len != 6) ERROR("Not a midi file");
    ui16 format,ntr,div;
    
    ioreadbe16(io,&format);
    ioreadbe16(io,&ntr);
    ioreadbe16(io,&div);
    midifile* mf = jsm_newmidi(ntr,div,512);
    ui8** tracks = (ui8**)malloc(mf->numtracks*sizeof(void*));
    jsmio** tio = (jsmio**)malloc(mf->numtracks*sizeof(void*));
    ui32* ltime = (ui32*)malloc(mf->numtracks*sizeof(ui32));
    ui8* lstatus = (ui8*)malloc(mf->numtracks*sizeof(ui8));
    jsmevent* curevt = (jsmevent*)malloc(mf->numtracks*sizeof(jsmevent));
    int i;
    for (i=0;i<mf->numtracks;i++) {
	ioread(io,tbuf,4);
	if (strcmp(tbuf,MIDI_THEADER)) ERROR("Not a midi file");
	ioreadbe32(io,&len);
	tracks[i] = malloc(len);
	ioread(io,tracks[i],len);
	tio[i]=jsm_io_read_mem(tracks[i],len);
	ltime[i]=0;
	lstatus[i]=0;
	readevent(tio[i],&curevt[i],&ltime[i],&lstatus[i],i);
	
    }
    mf->numevents=0;
    while (1) {
	ui32 mintime=NO_MORE_EVENTS;
	int mi=-1;
	for (i=0;i<mf->numtracks;i++) {
	    if (ltime[i] < mintime) {
		mintime=ltime[i];
		mi=i;
	    }
	}
	if (mi<0) break;
	jsm_addevent(mf,&curevt[mi]);
	readevent(tio[mi],&curevt[mi],&ltime[mi],&lstatus[mi],mi);
	
    }
    double mpt=500.0/(double)mf->divis;
    int ofsmid=0;
    int ofsmil=0;
    for (i=0;i<mf->numevents;i++) {
	jsmevent* cur = &mf->events[i];
	cur->millistime=((int)((cur->miditime-ofsmid)*mpt))+ofsmil;
	if (cur->status==0xFF && cur->metanum==81) {
	    ofsmid=cur->miditime;
	    ofsmil=cur->millistime;
	    mpt=((cur->pdata[0]<<16) | (cur->pdata[1]<<8) | cur->pdata[2])/1000.0/(double)mf->divis;
	}
	
    }
    jsm_setcapacity(mf,mf->numevents);
    if (autoclose) ioclose(io);
    return mf;
}
static int compare_time(const void* a, const void* b) {
    int atime=((jsmevent*)a)->miditime;
    int btime=((jsmevent*)b)->miditime;
    if (atime==btime) {
	return a-b; // this should keep the sort stable
    }
    return atime-btime;
}
void jsm_sortevents(midifile* mf) {
    qsort(mf->events,mf->numevents,sizeof(jsmevent),compare_time);
}
void jsm_writemidi(jsmio* io, int autoclose, midifile* mf) {
    iowrite(io,MIDI_HEADER,4);
    int dummy=6;
    iowritebe32(io,&dummy);
    dummy=1;
    iowritebe16(io,&dummy);
    iowritebe16(io,&mf->numtracks);
    iowritebe16(io,&mf->divis);
    // This does no sort, so make sure to call jsm_sortevents!
    int i,j;
    for (i=0;i<mf->numtracks;i++) {
	ui8* ctrack=malloc(512);
	ui32 clen=0;
	ui32 ltime=0;
	ui8 lstatus=0;
	jsmio* tio=jsm_io_write_mem(&ctrack,&clen,512);
	for (j=0;j<mf->numevents;j++) {
	    jsmevent* cur=&mf->events[j];
	    if (cur->track != i) continue;
	    writeevent(tio,cur,&ltime,&lstatus);
	}
	ioclose(tio);
	iowrite(io,MIDI_THEADER,4);
	iowritebe32(io,&clen);
	iowrite(io,ctrack,clen);
	free(ctrack);
    }
    
    if (autoclose) ioclose(io);
}

int jsm_get_channel(jsmevent* evt) {
    return evt->status&0xF;
}
